﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
Imports System.Drawing
Imports System.Drawing.Imaging
Public Class MainForm

#Region "窗体级别的变量和常量"

    ' 为进行恢复而维护的原始图像。
    Private originalImage As Image = Nothing
    ' 正在更新和编辑的当前图像。
    Private currentImage As Image = Nothing
    ' 为撤消最近的更改而存储的图像。
    Private undoImage As Image = Nothing
    ' 用于确定当前缩放因子的值。
    Private zoomFactor As Double = 1
    ' 屏幕上的图像的宽度（以像素为单位）。
    Private screenImageWidth As Integer = 0
    ' 屏幕上的图像的高度（以像素为单位）。
    Private screenImageHeight As Integer = 0
    ' 图像上鼠标的当前 X 值。
    Private X As Integer = 0
    ' 图像上鼠标的当前 Y 值。
    Private Y As Integer = 0

    ' 用于创建缩略图的当前图像大小的百分比。
    Const thumbnailFactor As Double = 0.15

#End Region

#Region "GUI 方法"

    ''' <summary>
    ''' 鼠标在 MainImage 上移动时更新 X 和 Y 坐标。
    ''' </summary>
    Private Sub MainImage_MouseMove(ByVal sender As Object, _
        ByVal e As System.Windows.Forms.MouseEventArgs) Handles MainImage.MouseMove

        ' 确定边界。
        Dim left As Integer = CInt((MainImage.Width - screenImageWidth) / 2)
        Dim top As Integer = CInt((MainImage.Height - screenImageHeight) / 2)
        Dim right As Integer = left + screenImageWidth
        Dim bottom As Integer = top + screenImageHeight

        ' 确定鼠标相对于图像的位置。
        If e.X < left Then
            X = 0
            XLabel.Text = "X: 0"
        ElseIf e.X > right Then
            X = CInt(screenImageWidth / zoomFactor)
            XLabel.Text = "X: " & X.ToString()
        Else
            X = CInt((e.X - left) / zoomFactor)
            XLabel.Text = "X: " + X.ToString()
        End If

        If e.Y < top Then
            Y = 0
            YLabel.Text = "Y: 0"
        ElseIf e.Y > bottom Then
            Y = CInt(screenImageHeight / zoomFactor)
            YLabel.Text = "Y: " & Y.ToString()
        Else
            Y = CInt((e.Y - top) / zoomFactor)
            YLabel.Text = "Y: " + Y.ToString()
        End If

    End Sub

    ''' <summary>
    ''' 退出应用程序
    ''' </summary> 
    Private Sub ExitApplication_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)
        Application.Exit()
    End Sub

#End Region

#Region "缩放方法"

    ''' <summary>
    ''' 缩放至 25%
    ''' </summary>
    Private Sub Zoom25_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Zoom25.Click
        Zoom(0.25)
    End Sub

    ''' <summary>
    ''' 缩放至 50%
    ''' </summary>
    Private Sub Zoom50_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Zoom50.Click
        Zoom(0.5)
    End Sub

    ''' <summary>
    ''' 缩放至 100%
    ''' </summary>
    Private Sub Zoom100_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Zoom100.Click
        Zoom(1)
    End Sub

    ''' <summary>
    ''' 缩放至 150%
    ''' </summary>
    Private Sub Zoom150_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Zoom150.Click
        Zoom(1.5)
    End Sub

    ''' <summary>
    ''' 缩放至 200%
    ''' </summary>
    Private Sub Zoom200_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Zoom200.Click
        Zoom(2)
    End Sub

    ''' <summary>
    ''' 更新缩放菜单以便让相应的选项处于选定状态
    ''' </summary>
    ''' <remarks></remarks>
    Private Sub UpdateZoomMenu()

        Zoom25.Checked = False
        Zoom50.Checked = False
        Zoom100.Checked = False
        Zoom150.Checked = False
        Zoom200.Checked = False

        Select Case zoomFactor
            Case 0.25
                Zoom25.Checked = True
                Exit Select
            Case 0.5
                Zoom50.Checked = True
                Exit Select
            Case 1
                Zoom100.Checked = True
                Exit Select
            Case 1.5
                Zoom150.Checked = True
                Exit Select
            Case 2
                Zoom200.Checked = True
                Exit Select
        End Select

    End Sub

    ''' <summary>
    ''' 将图像缩放至当前图像的指定倍数。
    ''' </summary>
    Private Sub Zoom(ByVal factor As Double)

        ' 将缩放倍数保存在全局变量中。
        zoomFactor = factor

        ' 获取调整大小后的图像。
        Dim sourceBitmap As New Bitmap(currentImage)
        Dim destBitmap As New Bitmap(CInt(sourceBitmap.Width * factor), _
            CInt(sourceBitmap.Height * factor))
        Dim destGraphic As Graphics = Graphics.FromImage(destBitmap)

        destGraphic.DrawImage(sourceBitmap, 0, 0, destBitmap.Width + 1, _
            destBitmap.Height + 1)

        ' 将图像在屏幕上的大小保存在全局变量中。
        screenImageWidth = destBitmap.Width
        screenImageHeight = destBitmap.Height

        MainImage.Image = destBitmap

        ' 更新窗体上的缩放标签。
        ZoomLabel.Text = "Zoom: " & zoomFactor * 100 & "%"

        ' 更新缩放菜单选择。
        UpdateZoomMenu()

    End Sub

#End Region

#Region "图像绘制、转换和更新方法"

    ''' <summary>
    ''' 使用指定的 ColorMatrix 绘制图像
    ''' </summary>
    Private Function DrawAdjustedImage(ByVal cMatrix As ColorMatrix) As Boolean

        ' 更新撤消图像和菜单。
        undoImage = CType(currentImage.Clone(), Image)
        Undo.Enabled = True
        Try
            Dim bmp As New Bitmap(currentImage)
            Dim rc As New Rectangle(0, 0, _
                currentImage.Width, currentImage.Height)
            Dim graphicsObject As Graphics = _
                Graphics.FromImage(currentImage)

            ' 将 ColorMatrix 对象与 ImageAttributes 对象关联
            Dim imgattr As New ImageAttributes()
            imgattr.SetColorMatrix(cMatrix)

            '应用 ColorMatrix
            graphicsObject.DrawImage(bmp, rc, 0, 0, currentImage.Width, _
                currentImage.Height, GraphicsUnit.Pixel, imgattr)

            graphicsObject.Dispose()

            ' 对图像应用当前缩放倍数。
            Zoom(zoomFactor)

            Return True
        Catch
            Return False
        End Try

    End Function

    ''' <summary>
    ''' 将图像大小调整为当前大小的某个百分比。
    ''' </summary>
    Private Sub ResizeImage(ByVal percent As Double)

        ' 更新撤消图像和菜单。
        undoImage = CType(currentImage.Clone(), Image)
        Undo.Enabled = True

        ' 调整图像的大小。
        Dim sourceBitmap As New Bitmap(currentImage)
        Dim destBitmap As New Bitmap(CInt(sourceBitmap.Width * percent / 100), CInt(sourceBitmap.Height * percent / 100))
        Dim destGraphic As Graphics = Graphics.FromImage(destBitmap)
        destGraphic.DrawImage(sourceBitmap, 0, 0, destBitmap.Width + 1, destBitmap.Height + 1)
        currentImage = destBitmap

        ' 缩放到当前缩放设置。
        Zoom(zoomFactor)

        ' 更新窗体上的宽度和高度标签。
        UpdateWidthandHeight()

    End Sub

    ''' <summary>
    ''' 使当前图像反相
    ''' </summary>
    Private Sub DrawNegativeImage()

        ' 创建反相图像的颜色矩阵。
        Dim cMatrix As ColorMatrix = New ColorMatrix(New Single()() _
                           {New Single() {-1, 0, 0, 0, 0}, _
                            New Single() {0, -1, 0, 0, 0}, _
                            New Single() {0, 0, -1, 0, 0}, _
                            New Single() {0, 0, 0, 1, 0}, _
                            New Single() {0, 0, 0, 0, 1}})

        ' 使用该颜色矩阵绘制图像。
        DrawAdjustedImage(cMatrix)

    End Sub

    ''' <summary>
    ''' 将当前图像转换为灰度图像
    ''' </summary>
    Public Function ConvertToGrayScale() As Boolean

        ' 创建灰度图像的颜色矩阵。
        Dim cMatrix As ColorMatrix = New ColorMatrix(New Single()() _
                               {New Single() {0.299, 0.299, 0.299, 0, 0}, _
                                New Single() {0.587, 0.587, 0.587, 0, 0}, _
                                New Single() {0.114, 0.114, 0.114, 0, 0}, _
                                New Single() {0, 0, 0, 1, 0}, _
                                New Single() {0, 0, 0, 0, 1}})

        ' 使用该颜色矩阵绘制图像。
        DrawAdjustedImage(cMatrix)

    End Function

    ''' <summary>
    ''' 使当前图像反相
    ''' </summary>
    Private Sub Negative_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Negative.Click
        DrawNegativeImage()
    End Sub

    ''' <summary>
    ''' 将当前图像转换为灰度图像。
    ''' </summary>
    Private Sub Grayscale_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Grayscale.Click
        ConvertToGrayScale()
    End Sub

    ''' <summary>
    ''' 调整图像大小
    ''' </summary>
    Private Sub ResizeButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ResizeButton.Click

        Dim ValidationMsg As String = _
            txtPercent.Tag.ToString & " must be a positive integer"

        ' 验证文本框中的输入。
        With txtPercent

            ValidationMsg &= "."

            If .Text.Trim = "" Then
                RejectTextBox(ValidationMsg, txtPercent)
                Exit Sub
            End If

            If Not IsNumeric(.Text.Trim) Then
                RejectTextBox(ValidationMsg, txtPercent)
                Exit Sub
            Else
                Dim integerValue As Integer
                Try
                    integerValue = CInt(.Text.Trim)
                    If integerValue <= 0 Then
                        RejectTextBox(ValidationMsg, txtPercent)
                        Exit Sub
                    End If
                Catch Exp As Exception
                    RejectTextBox(ValidationMsg, txtPercent)
                    Exit Sub
                End Try
            End If
        End With

        ' 调用 ResizeImage 方法。
        ResizeImage(CDbl(txtPercent.Text))

    End Sub

    ''' <summary>
    ''' 将裁切值转换为适合当前图像的值。
    ''' </summary>
    Private Sub TranslateCropValues()

        If CInt(LeftInput.Text) > currentImage.Width Then
            LeftInput.Text = CStr(currentImage.Width)
        End If

        If CInt(TopInput.Text) > currentImage.Height Then
            TopInput.Text = CStr(currentImage.Height)
        End If

        If CInt(LeftInput.Text) + CInt(WidthInput.Text) > currentImage.Width Then
            WidthInput.Text = CStr(currentImage.Width - CInt(LeftInput.Text))
        End If

        If CInt(TopInput.Text) + CInt(HeightInput.Text) > currentImage.Height Then
            HeightInput.Text = CStr(currentImage.Height - CInt(TopInput.Text))
        End If

    End Sub

    ''' <summary>
    ''' 按照裁切值裁切图像。
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    ''' <remarks></remarks>
    Private Sub CropButton_Click(ByVal sender As System.Object, _
        ByVal e As System.EventArgs) Handles CropButton.Click

        If IsValidCroppingInput() Then
            undoImage = CType(currentImage.Clone(), Image)
            Undo.Enabled = True

            TranslateCropValues()

            Dim recSource As New Rectangle( _
                CInt(LeftInput.Text), CInt(TopInput.Text), _
                CInt(WidthInput.Text), CInt(HeightInput.Text))

            Dim bmpCropped As New Bitmap( _
                CInt(WidthInput.Text), CInt(HeightInput.Text))

            Dim grBitmap As Graphics = Graphics.FromImage(bmpCropped)

            grBitmap.DrawImage(currentImage, 0, 0, _
                recSource, GraphicsUnit.Pixel)

            currentImage = bmpCropped
            Zoom(zoomFactor)
        End If

    End Sub

    ''' <summary>
    ''' 确定裁切值是否有效
    ''' </summary>
    Private Function IsValidCroppingInput() As Boolean

        ' 迭代文本框
        For Each txt As TextBox In New TextBox() _
            {TopInput, LeftInput, WidthInput, HeightInput}
            Dim ValidationMsg As String = _
                txt.Tag.ToString & " must be a positive integer"
            With txt
                If (txt Is LeftInput) Or (txt Is TopInput) Then
                    ValidationMsg &= " or zero."
                Else
                    ValidationMsg &= "."
                End If

                ' 如果不存在输入。
                If .Text.Trim = "" Then
                    RejectTextBox(ValidationMsg, txt)
                    Return False
                End If

                ' 如果存在非数字输入。
                If Not IsNumeric(.Text.Trim) Then
                    RejectTextBox(ValidationMsg, txt)
                    Return False
                Else
                    Dim integerValue As Integer
                    ' 确定输入是否为正。
                    Try
                        integerValue = CInt(.Text.Trim)
                        If (txt Is LeftInput) Or (txt Is TopInput) Then
                            If integerValue < 0 Then
                                RejectTextBox(ValidationMsg, txt)
                                Return False
                            End If
                        ElseIf integerValue <= 0 Then
                            RejectTextBox(ValidationMsg, txt)
                            Return False
                        End If
                    Catch Exp As Exception
                        RejectTextBox(ValidationMsg, txt)
                        Return False
                    End Try
                End If
            End With
        Next

        Return True

    End Function

    Private Sub RejectTextBox(ByVal message As String, ByVal txt As TextBox)
        MessageBox.Show(message, Me.Text, MessageBoxButtons.OK, MessageBoxIcon.Error)
        txt.SelectAll()
        txt.Focus()
    End Sub

    ''' <summary>
    ''' 绘制一个框以表明裁切位置。
    ''' </summary>
    Private Sub TestButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles TestButton.Click

        If IsValidCroppingInput() Then

            ' 重绘图像以移除以前的裁切框。
            MainImage.Refresh()

            Dim left As Integer = CInt((MainImage.Width - screenImageWidth) / 2)
            Dim top As Integer = CInt((MainImage.Height - screenImageHeight) / 2)

            TranslateCropValues()

            ' 绘制一个红色矩形以表明图像将被裁切的位置。
            Dim recCropBox As New Rectangle(CInt(CDbl(LeftInput.Text) * zoomFactor + left - 1), _
                CInt(CDbl(TopInput.Text) * zoomFactor + top - 1), CInt(CDbl(WidthInput.Text) * zoomFactor), _
                    CInt(CDbl(HeightInput.Text) * zoomFactor))
            MainImage.CreateGraphics.DrawRectangle(Pens.Red, recCropBox)
        End If

    End Sub


    ''' <summary>
    ''' 更新窗体上的宽度和高度标签。
    ''' </summary>
    Private Sub UpdateWidthandHeight()

        WidthLabel.Text = "Width: " & currentImage.Width
        HeightLabel.Text = "Height: " & currentImage.Height

    End Sub

#End Region

#Region "旋转和翻转方法"

    ''' <summary>
    ''' 旋转或翻转当前图像。
    ''' </summary>
    Private Sub RotateFlip(ByVal degrees As Integer)

        ' 更新撤消图像和菜单。
        undoImage = CType(currentImage.Clone(), Image)
        Undo.Enabled = True
        Select Case degrees
            Case 0
                currentImage.RotateFlip(RotateFlipType.RotateNoneFlipX)
            Case 1
                currentImage.RotateFlip(RotateFlipType.RotateNoneFlipY)
            Case 90
                currentImage.RotateFlip(RotateFlipType.Rotate90FlipNone)
                Exit Select
            Case 180
                currentImage.RotateFlip(RotateFlipType.Rotate180FlipNone)
                Exit Select
            Case 270
                currentImage.RotateFlip(RotateFlipType.Rotate270FlipNone)
            Case Else
                Exit Select
        End Select

        ' 使用缩放设置显示当前图像。
        Zoom(zoomFactor)

    End Sub

    ''' <summary>
    ''' 将当前图像旋转 90 度。
    ''' </summary>
    Private Sub Rotate90_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Rotate90.Click
        RotateFlip(90)
    End Sub

    ''' <summary>
    ''' 将图像旋转 180 度
    ''' </summary>
    Private Sub Rotate180_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Rotate180.Click
        RotateFlip(180)
    End Sub

    ''' <summary>
    ''' 将图像旋转 270 度
    ''' </summary>
    Private Sub Rotate270_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Rotate270.Click
        RotateFlip(270)
    End Sub

    ''' <summary>
    ''' 沿 Y 轴翻转当前图像
    ''' </summary>
    Private Sub HorizontalFlip_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles HorizontalFlip.Click
        RotateFlip(0)
        Zoom(zoomFactor)
    End Sub

    ''' <summary>
    ''' 沿 X 轴翻转当前图像。
    ''' </summary>
    Private Sub VerticalFlip_Click(ByVal sender As System.Object, _
     ByVal e As System.EventArgs) Handles VerticalFlip.Click
        RotateFlip(1)
        Zoom(zoomFactor)
    End Sub

#End Region

#Region "恢复和撤消方法"

    ''' <summary>
    ''' 将原始图像还原为其磁盘上的表示形式
    ''' </summary>
    Private Sub Revert_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Revert.Click
        undoImage = CType(currentImage.Clone(), Image)
        currentImage = CType(originalImage.Clone(), Image)
        Zoom(zoomFactor)
    End Sub

    ''' <summary>
    ''' 使用上次更改中的撤消图像还原图像。
    ''' </summary>
    Private Sub Undo_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Undo.Click
        currentImage = CType(undoImage.Clone(), Image)
        Zoom(zoomFactor)

        ' 将撤消菜单更新为禁用状态。
        Undo.Enabled = False
    End Sub

#End Region

#Region "打开和保存方法"

    ''' <summary>
    ''' 从文件打开图像并更新窗体。
    ''' </summary>
    Private Sub OpenImage_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles OpenImage.Click

        If OpenFileDialog1.ShowDialog = Windows.Forms.DialogResult.OK Then
            ' 打开图像
            originalImage = Image.FromFile(OpenFileDialog1.FileName)
            currentImage = CType(originalImage.Clone(), Image)

            ' 确定初始图像视图的适当缩放倍数。
            If currentImage.Width / 2 > MainImage.Width Or _
             currentImage.Height / 2 > MainImage.Height Then
                Zoom(0.25)
            ElseIf currentImage.Width > MainImage.Width Or _
             currentImage.Height > MainImage.Height Then
                Zoom(0.5)
            ElseIf currentImage.Width * 2 < MainImage.Width _
             And currentImage.Height * 2 < MainImage.Height Then
                Zoom(2)
            ElseIf currentImage.Width * 2 > MainImage.Width _
             And currentImage.Height * 2 > MainImage.Height Then
                Zoom(2)
            ElseIf currentImage.Width * 1.5 < MainImage.Width _
             And currentImage.Height * 2 < MainImage.Height Then
                Zoom(1.5)
            Else
                Zoom(1)
            End If

            ' 针对新的图像更新窗体。
            UpdateWidthandHeight()
            Undo.Enabled = False
            ImageToolStripMenuItem.Visible = True
            EditMenu.Visible = True
            SaveThumbnailAs.Enabled = True
            SaveImageAs.Enabled = True
            Resizing.Enabled = True
            Cropping.Enabled = True
            ImageInfo.Enabled = True
            MainImage.Enabled = True

        End If

    End Sub

    ''' <summary>
    ''' 将图像保存到所选的文件。
    ''' </summary>
    Private Sub SaveImageAs_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SaveImageAs.Click
        If SaveFileDialog1.ShowDialog() = Windows.Forms.DialogResult.OK Then
            currentImage.Save(SaveFileDialog1.FileName, GetImageFormat())
        End If
    End Sub

    ''' <summary>
    ''' 将当前图像保存为缩略图
    ''' </summary>
    Private Sub SaveThumbnailAs_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SaveThumbnailAs.Click

        Dim sourceBitmap As New Bitmap(currentImage)
        Dim destBitmap As New Bitmap( _
         CInt(sourceBitmap.Width * thumbnailFactor), _
          CInt(sourceBitmap.Height * thumbnailFactor))

        Dim destGraphic As Graphics = Graphics.FromImage(destBitmap)

        destGraphic.DrawImage(sourceBitmap, 0, 0, _
         destBitmap.Width + 1, destBitmap.Height + 1)

        If SaveFileDialog1.ShowDialog() = Windows.Forms.DialogResult.OK Then
            destBitmap.Save(SaveFileDialog1.FileName, GetImageFormat())
        End If

    End Sub

    ''' <summary>
    ''' 从保存对话框获取图像格式。
    ''' </summary>
    Private Function GetImageFormat() As ImageFormat
        Select Case SaveFileDialog1.FilterIndex
            Case 1
                Return ImageFormat.Bmp
            Case 2
                Return ImageFormat.Jpeg
            Case 3
                Return ImageFormat.Gif
            Case Else
                Return ImageFormat.Tiff
        End Select
    End Function

    Private Sub ExitApplication_Click_1(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ExitApplication.Click
        Me.Close()
    End Sub

#End Region

End Class
